<?php
/**
 * General purpose helper functionality.
 *
 * @package    Sms_Plugin_Com
 * @subpackage Sms_Plugin_Com/includes/helper
 * @link       https://www.sms-plugin.com
 * @author     sms-plugin.com <info@sms-plugin.com>
 * @since 1.0.0
 */

/**
 * Defines an abstract class with helper functions
 * that are used throughout the plugin
 *
 * @package    Sms_Plugin_Com
 * @subpackage Sms_Plugin_Com/includes/helper
 * @author     sms-plugin.com <info@sms-plugin.com>
 * @since 1.0.0
 */
abstract class Sms_Plugin_Com_Helper_Base {
  /**
   * Order statuses array
   *
   * @var array
   * @since 1.0.0
   */
  public $order_statuses = null;

  /**
   * Class constructor
   *
   * @access protected
   * @since 1.0.0
   */
  protected function __construct() {
    $this->order_statuses = array(  'pending' => __( 'Pending', 'sms_plugin_com' ),
                                    'processing' => __('Processing', 'sms_plugin_com'),
                                    'on-hold' => __('On-hold', 'sms_plugin_com'),
                                    'completed' => __('Completed', 'sms_plugin_com'),
                                    'cancelled' => __('Cancelled', 'sms_plugin_com'),
                                    'refunded' => __('Refunded', 'sms_plugin_com'),
                                    'failed' => __('Failed', 'sms_plugin_com')  );
  }

  /**
   * This function is used to prepare the sms messages
   *
   * @since 1.0.0
   * @access public
   * @param  string $message the message text
   * @param  array $data the data to be inserted to the message
   * @return string the message
   */
  public function prepare_message( $message, $data ) {
      $message = str_replace( array_keys($data), $data, $message );
      return $message;
  }

  /**
   * This function returns whether admin notifications are enabled
   *
   * @since 1.0.0
   * @access public
   * @return bool whether admin notifications are enabled
   */
  public function is_admin_notify_enabled() {
    $admin_notify = get_option( 'sms_plugin_com' )[ 'admin-notify' ];
    return  $admin_notify == 'true' ? true : false;
  }

  /**
   * This function returns whether admin notifications
   * on new account are enabled
   *
   * @since 1.0.0
   * @access public
   * @return bool
   */
  public function is_admin_notify_new_account_enabled() {
    $admin_notify_new_account  = get_option( 'sms_plugin_com' )[ 'admin-notify-new-account' ];
    return $admin_notify_new_account == 'true' ? true : false;
  }

  /**
   * This function returns whether admin notifications
   * on new order are enabled
   *
   * @since 1.0.0
   * @access public
   * @return bool
   */
  public function is_admin_notify_new_oder_enabled() {
    $admin_notify_new_order  = get_option( 'sms_plugin_com' )[ 'admin-notify-customer-checkout' ];
    return $admin_notify_new_order == 'true' ? true : false;
  }

  /**
   * This function returns whether admin notifications
   * on order status change are enabled
   *
   * @since 1.0.0
   * @access public
   * @return bool
   */
  public function is_admin_notify_order_status_change_enabled() {
    $admin_notify_order_status_change  = get_option( 'sms_plugin_com' )[ 'admin-notify-order-status' ];
    return $admin_notify_order_status_change == 'true' ? true : false;
  }

  /**
   * This function returns admin mobile phone number
   * from plugin settings
   *
   * @since 1.0.0
   * @access public
   * @return string
   */
  public function get_admin_mobile() {
    $admin_mobile = get_option( 'sms_plugin_com' )[ 'admin-mobile' ];
    return $admin_mobile;
  }

  /**
   * This function returns the admin notification encoding
   * from plugin settings
   *
   * @since 1.0.0
   * @access public
   * @return string
   */
  public function get_admin_notify_encoding() {
    $admin_notify_encoding = get_option( 'sms_plugin_com' )[ 'admin-notify-encoding' ];
    return $admin_notify_encoding ===  null ? '1' : $admin_notify_encoding;
  }

  /**
   * This function returns admin user register message
   *
   * @since 1.0.0
   * @access public
   * @return string
   */
  public function get_admin_user_register_message( $set_names ) {
    $message = get_option( 'sms_plugin_com' )[ 'admin-notify-new-account-sms-text' ];
    if( null === $message ) {
      $message =  __( "New account registration login: {user_login}, email: {user_email}", 'sms_plugin_com' );
      if ( $set_names ) {
        $message =  __( "New account registration: {first_name} {last_name}, email: {user_email}", 'sms_plugin_com' );
      }
    }
    return $message;
  }

  /**
   * This function returns whether customer notifications
   * on new order are enabled
   *
   * @since 1.0.0
   * @access public
   * @return bool
   */
  public function is_customer_notify_new_order_enabled() {
    $customer_notify = get_option( 'sms_plugin_com' )[ 'new-order-send-sms' ];
    return  $customer_notify == 'true' ? true : false;
  }

  /**
   * This function returns the new order sender id
   * from plugin settings
   *
   * @since 1.0.0
   * @access public
   * @return string
   */
  public function get_new_order_sender() {
    $new_order_sender = get_option( 'sms_plugin_com' )[ 'new-order-sender' ];
    return $new_order_sender === null ? $this->get_store_name() : $new_order_sender;
  }

  /**
   * This function returns the new order sms encoding
   * from plugin settings
   *
   * @since 1.0.0
   * @access public
   * @return string
   */
  public function get_new_order_encoding() {
    $new_order_encoding = get_option( 'sms_plugin_com' )[ 'new-order-encoding' ];
    return $new_order_encoding ===  null ? '1' : $new_order_encoding;
  }

  /**
   * This function returns the new order customer message
   * from plugin settings
   *
   * @since 1.0.0
   * @access public
   * @return string
   */
  public function get_new_order_customer_message() {
    $message = get_option( 'sms_plugin_com' )[ 'new-order-sms-text' ];
    if ( null === $message ) {
      $message = __( "Thank you for placing your order with www.yourdomain.com. Your order ID is {order_id} and is currently being processed. Your Store.", 'sms_plugin_com' );
    }
    return $message;
  }

  /**
   * This function returns the new order admin message
   *
   * @since 1.0.0
   * @access public
   * @return string
   */
  public function get_new_order_admin_message() {
    $message = get_option( 'sms_plugin_com' )[ 'admin-notify-customer-checkout-sms-text' ];
    if ( null === $message ) {
        $message = __( "{first_name} {last_name} (email: {user_email}) has placed an order (id: {order_id}) with amount {total} {currency_code}", 'sms_plugin_com' );
    }
    return $message;
  }

  /**
   * This function returns whether customer notifications
   * on order status change are enabled
   *
   * @since 1.0.0
   * @access public
   * @return bool
   */
  public function is_customer_notify_order_status_change_enabled() {
    $customer_notify = get_option( 'sms_plugin_com' )[ 'order-status-send-sms' ];
    return  $customer_notify == 'true' ? true : false;
  }

  /**
   * This function returns the order status change sender id
   * from plugin settings
   *
   * @since 1.0.0
   * @access public
   * @return string
   */
  public function get_order_status_change_sender() {
    $order_status_change_sender = get_option( 'sms_plugin_com' )[ 'new-order-sender' ];
    return $order_status_change_sender === null ? $this->get_store_name() : $order_status_change_sender;
  }

  /**
   * This function returns the order status change sms encoding
   * from plugin settings
   *
   * @since 1.0.0
   * @access public
   * @return string
   */
  public function get_order_status_change_encoding() {
    $order_status_change_encoding = get_option( 'sms_plugin_com' )[ 'order-status-encoding' ];
    return $order_status_change_encoding ===  null ? '1' : $order_status_change_encoding;
  }

  /**
   * This function returns the order status change customer message
   * from plugin settings
   *
   * @since 1.0.0
   * @access public
   * @return string
   */
  public function get_order_status_change_customer_message() {
    $message = get_option( 'sms_plugin_com' )[ 'order-status-sms-text' ];
    if ( null === $message ) {
      $message = __ ( "Your order # {order_id} has been marked as {order_status}. Thanks for shopping at www.yourdomain.com", 'sms_plugin_com');
    }
    return $message;
  }

  /**
   * This function returns the order status change admin message
   *
   * @since 1.0.0
   * @access public
   * @return string
   */
  public function get_order_status_change_admin_message() {
    $message = get_option( 'sms_plugin_com' )[ 'admin-notify-order-status-sms-text' ];
    if( null === $message ) {
      $message = __( "{first_name} {last_name}'s (email: {user_email}) order (id: {order_id}) has changed to {order_status}", 'sms_plugin_com' );
    }
    return $message;
  }

  /**
   * This function returns the store/blog name
   * from settings
   *
   * @since 1.0.0
   * @access public
   * @return string
   */
  public function get_store_name() {
    $store_name = get_bloginfo( 'name' );
    return $store_name;
  }

  /**
   * This function prepares the user data
   * to be inserted in the sms message
   *
   * @since 1.0.0
   * @access public
   * @return array
   */
  public function get_user_data( $user_id ) {
    $user = get_userdata(  $user_id  );
    $user_data = [
            '{user_login}' => $user->user_login,
            '{user_email}' => $user->user_email
           ];

    if( isset( $_POST[ 'billing_first_name' ] ) ) {
      $user_data[ '{first_name}' ] = $_POST[ 'billing_first_name' ];
    }

    if( isset( $_POST[ 'billing_last_name' ] ) ) {
      $user_data[ '{last_name}' ] = $_POST[ 'billing_last_name' ];
    }

    return $user_data;
  }

  /**
   * This function returns customer's mobile phone number
   * from order data
   *
   * @since 1.0.0
   * @access public
   * @return string
   */
  public function get_customer_mobile( $order_id ) {
    $order = new WC_Order( $order_id );
    $customer_mobile = $order->billing_phone;
    return $customer_mobile;
  }

  /**
   * This function prepares the order data
   * to be inserted in the sms message
   *
   * @since 1.0.0
   * @access public
   * @return array
   */
  public function get_order_data( $order_id ) {
    $order = new WC_Order( $order_id );
    $order_data = [
            '{order_id}' => $order->get_order_number(),
            '{first_name}' => $order->billing_first_name,
            '{last_name}' => $order->billing_last_name,
            '{user_email}' => $order->billing_email,
            '{total}' => $order->get_total(),
            '{currency_code}' => $order->get_order_currency(),
            '{order_status}' => __( $this->order_statuses[$order->get_status()], 'sms_plugin_com' )
        ];
    return $order_data;
  }

  /**
	 * Logging function
	 *
	 * @since 	1.0.0
	 * @access	public
	 * @param   string $text the text to log.
	 */
	public function log( $text ) {
		global $wp_filesystem;

		if ( empty( $wp_filesystem ) ) {
			  require_once( ABSPATH . '/wp-admin/includes/file.php' );
			  WP_Filesystem();
		}
		$time = date( 'd/m/Y H:i:s', time() + 10800 );
		$content = $wp_filesystem->get_contents(self::FILE_PATH);

		if ( is_array( $text ) || is_object( $text ) ) {
        $content .= "#$time\t". print_r($text, true) . "\r\n";
    } else {
        $content .= "#$time\t$text\r\n";
    }
		$wp_filesystem->put_contents( self::FILE_PATH, $content, FS_CHMOD_FILE );
	}
}
