<?php
/**
 * The woocommerce-specific functionality of the plugin.
 *
 * @package    Sms_Plugin_Com
 * @subpackage Sms_Plugin_Com/woocommerce
 * @link       https://www.sms-plugin.com
 * @author     sms-plugin.com <info@sms-plugin.com>
 * @since 1.0.0
 */

/**
 * Abstract class that defines the actions for WooCommerce hooks
 *
 * @package    Sms_Plugin_Com
 * @subpackage Sms_Plugin_Com/woocommerce
 * @author     sms-plugin.com <yannis@tern.gr>
 * @since 1.0.0
 */
abstract class Sms_Plugin_Com_Wc_Base {
  /**
   * Variable to assign instance of Sms_Plugin_Com_Helper class
   *
   * @since 1.0.0
   */
  private static $helper = null;

  protected function __construct() {
    self::$helper = Sms_Plugin_Com_Helper::instance();
  }
  /**
   * This function is an action for user registration hook
   *
   * @since 1.0.0
   * @access public
   * @param  string $user_id the user id
   */
  public function user_register(  $user_id  ) {
    if( self::$helper->is_admin_notify_enabled() && self::$helper->is_admin_notify_new_account_enabled() ) {
      $store_name = self::$helper->get_store_name();
      $user_data = self::$helper->get_user_data(  $user_id  );
      $message = self::$helper->get_admin_user_register_message( false );
      $admin_notify_encoding = self::$helper->get_admin_notify_encoding();
      if ( isset( $user_data[ '{first_name}' ] ) && isset( $user_data[ '{last_name}' ] ) ) {
        $message = self::$helper->get_admin_user_register_message( true );
      }
      $message = self::$helper->prepare_message( $message, $user_data );
      $admin_mobile = self::$helper->get_admin_mobile();
      if( null !== $admin_mobile ) {
          Sms_Plugin_Com_Api::instance()->send_sms( $store_name, $admin_mobile, $message, $admin_notify_encoding );
      }
    }
  }

  /**
   * This function is an action for customer checkout hook
   *
   * @since 1.0.0
   * @access public
   * @param  string $order_id the order id
   */
  public function order_checkout( $order_id ) {
    $data = self::$helper->get_order_data( $order_id );
    $store_name = self::$helper->get_store_name();
    $data['{store_name}'] = $store_name;
    $new_order_encoding = self::$helper->get_new_order_encoding();
    $admin_notify_encoding = self::$helper->get_admin_notify_encoding();

    // Notify admin on new order
    if( self::$helper->is_admin_notify_enabled() && self::$helper->is_admin_notify_new_oder_enabled() ) {
      $message = self::$helper->get_new_order_admin_message();
      $message = self::$helper->prepare_message( $message, $data );
      $admin_mobile = self::$helper->get_admin_mobile();
      if( null !== $admin_mobile ) {
          Sms_Plugin_Com_Api::instance()->send_sms( $store_name, $admin_mobile, $message, $admin_notify_encoding, $order_id );
      }
    }

    // Notify customer on checkout
    if(self::$helper->is_customer_notify_new_order_enabled() ) {
      $message = self::$helper->get_new_order_customer_message();
      $message = self::$helper->prepare_message( $message, $data );
      $new_order_sender = self::$helper->get_new_order_sender();
      $customer_mobile = self::$helper->get_customer_mobile( $order_id );
      if( null !== $customer_mobile ) {
          Sms_Plugin_Com_Api::instance()->send_sms( $new_order_sender, $customer_mobile, $message, $new_order_encoding, $order_id );
      }
    }
  }

  /**
   * This function is an action for order status changes
   *
   * @since 1.0.0
   * @access public
   * @param  string $order_id the order id
   */
  public function order_status_changed( $order_id ) {
    $data = self::$helper->get_order_data( $order_id );
    $data['{store_name}'] = self::$helper->get_store_name();
    $order_status_encoding = self::$helper->get_order_status_change_encoding();
    $admin_notify_encoding = self::$helper->get_admin_notify_encoding();

    // Notify admin on order status change
    if( self::$helper->is_admin_notify_enabled() && self::$helper->is_admin_notify_order_status_change_enabled() ) {
      $message = self::$helper->get_order_status_change_admin_message();
      $message = self::$helper->prepare_message( $message, $data );
      $admin_mobile = self::$helper->get_admin_mobile();
      if( null !== $admin_mobile ) {
          Sms_Plugin_Com_Api::instance()->send_sms( $data['{store_name}'], $admin_mobile, $message, $admin_notify_encoding, $order_id );
      }
    }

    // Notify customer on order status change
    if(self::$helper->is_customer_notify_order_status_change_enabled() ) {
      $message = self::$helper->get_order_status_change_customer_message();
      $message = self::$helper->prepare_message( $message, $data );
      $order_status_change_sender = self::$helper->get_order_status_change_sender();
      $customer_mobile = self::$helper->get_customer_mobile( $order_id );
      if( null !== $customer_mobile ) {
          Sms_Plugin_Com_Api::instance()->send_sms( $order_status_change_sender, $customer_mobile, $message, $order_status_encoding, $order_id );
      }
    }
  }

  /**
   * Deactivates plugin if WooCommerce is uninstalled or deactivated
   * and prints a message on admin panel
   *
   * @since 1.0.0
   * @access public
   */
  public function admin_notice_wc_not_installed() {
    /* Check transient, if available display notice */
    if( get_transient( 'wc-not-installed-or-deactivated-admin-notice' ) ){
        ?>
        <div class="error notice is-dismissible">
            <p><?php echo '<h3> SMS Plugin for WooCommerce</h3><h4>' . __('SMS Plugin for WooCommerce requires an active WooCommerce installation. Please install/activate WooCommerce!') . '</h4>' ?></p>
        </div>
        <?php
        /* Delete transient, only display this notice once. */
        delete_transient( 'wc-not-installed-or-deactivated-admin-notice' );
        deactivate_plugins( 'sms-plugin-com/sms-plugin-com.php' );
    }
  }
}
