/**
 * This js file contains several functions that are used
 * throughout the settings area af the plugin
 * @since    1.0.0
 */

 if (typeof(console) == 'object') {
     console.log('sms-plugin-com-functions.js script loaded');
 }

 /**
  * Function used to handle ajax  requests
  *
  * @param  string  type          POST|GET|PUT etc.
  * @param  array   params        array of parameters that contains e.g url and messages.
  * @param  string  data          the data to send.
  * @param  {Function} callback   the function to be called back after response arrives.
  * @param  {Function} next       function to execute next.
  * @since    1.0.0
  */
 function ajax_call(type, params, data, callback, next) {
    // AJAX POST to admin-ajax.php file in wp-admin directory
    jQuery.ajax({
       type: type,
       url: params.url,
       data: data,
       success: function(response) {
         response = JSON.parse(response);
         switch(response.status_code) {
           case 200:
              callback(response, params.message_200);
              break;
           case 401:
              callback(response, params.message_401);
              break;
           case 402:
              callback(response, params.message_402);
              break;
           case 500:
              callback(response, params.message_500);
              break;
         }
       }
     }).done(function() {
        if(undefined !== next) {
          next();
        }
     });
 }

/**
 * Function to get balance
 *
 * @param  string data          data to send.
 * @param  string element_id    id of html element that will display the balance.
 * @since    1.0.0
 */
function get_balance(data, element_id) {
  jQuery("#balance-div").slideUp(200);
  jQuery(".loader").slideDown(200);
   ajax_call('POST', sms_plugin_com_get_balance, data, function(response, message) {
     if(response.body === undefined) {
       jQuery(element_id).text('-');
       sms_plugin_tooltip('balance','before','balance-tooltip','right','large',message,'5000');
     } else {
       jQuery(element_id).text(response.body.balance);
       jQuery("#balance-div").slideDown(200);
       jQuery(".loader").slideUp(200);
     }
   });
 }

 /**
  * Function to download sms list in csv
  *
  * @param  string data          data to send.
  * @param  string element_id    id of html element that will display message.
  * @param  {function}  next    function to execute next
  * @since    1.0.0
  */
 function get_csv(data) {
    var params = sms_plugin_com_history;
    ajax_call('POST', params, data, function(response, message) {
      var headers = {
          smsplugin_id: 'ID'.replace(/,/g, ''), // remove commas to avoid errors
          send_tstamp: "Datetime Sent",
          sender_id: "Sender",
          to: "Recipient",
          message:  "Message",
          status: "Status",
          order_id: "Order"
      };
      var date = new Date();
      var dd = date.getDate();
      var mm = date.getMonth()+1;
      var yyyy = date.getFullYear();
      var title = 'sms-list-' + dd + '-' + mm + '-' + yyyy
      exportCSVFile(headers, response.body, title)
    });
  }

/**
 * Function to get/reset the API key
 *
 * @param  string data          data to send.
 * @param  string element_id    id of html element that will display the API key.
 * @param  {function}  next    function to execute next
 * @since    1.0.0
 */
function get_key(data, element_id, next) {
   var params = sms_plugin_com_get_key;
   if(data.reset === true) {
      params = sms_plugin_com_reset_key;
   }
   ajax_call('POST', params, data, function(response, message) {
       jQuery( element_id ).val( response.body.key ).fadeTo( 100, 0.1 ).fadeTo( 200, 1.0 );
       message = message.replace( '{$}', response.body.key );
       sms_plugin_tooltip( 'sms_plugin_com-api-key','before','api-key-tooltip','up','large',message,'5000' );
   }, next);
 }

 /**
  * Function to send sms
  *
  * @param  string      data     data to send.
  * @param  {function}  next    function to execute next
  * @since    1.0.0
  */
 function send_sms(data, next) {
    var params = sms_plugin_com_send_sms;
    ajax_call('POST', params, data, function(response, message) {
        message = message.replace( '{$}', response.body.remarks );
        switch(response.status_code) {
          case 200:
          case 402:
          case 500:
            sms_plugin_tooltip( 'balance','before','send-sms-tooltip','up','large',message,'5000' );
            break;
          case 401:
            sms_plugin_tooltip( 'sms_plugin_com-api-key','before','send-sms-tooltip','up','large',message,'5000' );
            break;
        }
    }, next);
  }

/**
 * This function handles the tooltip functionality
 *
 * @param  string element_id       [the id of the html element to put the tooltip]
 * @param  string position         [before|after the html element above]
 * @param  string tooltip_id       [the html id of the tooltip]
 * @param  string tooltip_position [up|left|right|down the position of the tooltip]
 * @param  string tooltip_size     [small|medium|large|xlarge|fit the size of the tooltip]
 * @param  string message          [the tooltip message]
 * @param  int    timeout          [hide after x ms , 0 for not hiding]
 * @since    1.0.0
 */
function sms_plugin_tooltip(element_id, position, tooltip_id, tooltip_position, tooltip_size, message, timeout)  {
  var html = '<div id="' + tooltip_id + '" data-balloon="' + message
              + '" data-balloon-pos="' + tooltip_position + '"'
              + '" data-balloon-length="' + tooltip_size + '"'
              + ' data-balloon-visible></div>';
  switch(position) {
    case "before":
      jQuery('#' + element_id).before(html);
      break;
    case "after":
      jQuery('#' + element_id).after(html);
      break;
  }
  setTimeout(function() {
    jQuery('#' + tooltip_id).remove();
  }, timeout);
}

/**
 * This function modifies the String prototype in order to add
 * check_sender_id function which checks the validity of the
 * sender id.
 *
 * @since  1.0.0
 */
function add_string_prototype() {
  String.prototype.check_sender_id = function(){
    var i = 0, c = 0, r ='',
        s = this,
        numericSenderId = "+1234567890",
        allowedChars = ":;+-.!0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ";
    for ( i = 0; i < s.length; i++ ){
        if ( allowedChars.indexOf( s.charAt( i ) ) > -1 ){
            r += s.charAt( i );
        }
    }
    //check if alpha or numeric and fix the length
    for ( i = 0; i < r.length; i++ ){
        if ( numericSenderId.indexOf( r.charAt( i ) ) > -1 ){
            c++;
        }
    }
    if ( c == r.length ){ // numeric
        return r.substring(0, 16);
    }else{ // alpha
        return r.substring(0, 11);
    }
  }
}

/**
 * Function to count characters depending on encoding
 *
 * @since  1.0.0
 * @return string length the length of the current_text
 */
function char_count(encoding, current_text) {
  var length = 0;
  if(encoding == '2') {
    length = current_text.length;
  } else if(encoding == '1') {
    for ( d = 0; d < current_text.length; d++) {
      n = current_text.charCodeAt(d);
      switch (n){
        case 12:
        case 91:
        case 92:
        case 93:
        case 94:
        case 123:
        case 124:
        case 125:
        case 126:
        case 8364:
          length += 2;
          break;
        default:
          length++;
      }
    }
  }
  return length;
}

/**
 * Set character and SMS counters
 *
 * @param string char_counter_id  id of character counter html element
 * @param string sms_counter_id   id of sms counter html element
 * @param number encoding_value   1 = GSM, 2 = UCS
 * @param string length           current text length
 * @since    1.0.0
 */
function set_counters(char_counter_id, sms_counter_id, encoding_value, length) {
  jQuery(char_counter_id).text(length);
  if(encoding_value === '2') {
    jQuery(sms_counter_id).text(Math.ceil(length/67));
  } else {
    jQuery(sms_counter_id).text(Math.ceil(length/157));
  }
}

/**
 * Function to dynamically add jQuery validation rules
 *
 * @param object rules JSON object of rules
 * @since    1.0.0
 */
function add_validation_rules(rules){
    for (var item in rules){
      jQuery('#'+item).rules('add', rules[item]);
    }
}

/**
 * Function to dynamically remove jQuery validation rules
 *
 * @param object rules JSON object of rules
 * @since    1.0.0
 */
function remove_validation_rules(rules){
    for (var item in rules){
       jQuery('#'+item).rules('remove');
    }
}

/**
 * Function to export sms list to csv file
 * @param  {Object} headers   The csv headers
 * @param  {Object} items     The list items
 * @param  {String} fileTitle The csv file title
 * @since    1.0.0
 */
function exportCSVFile(headers, items, fileTitle) {
    if (headers) {
      items.unshift(headers);
    }
    // Convert Object to JSON
    var jsonObject = JSON.stringify(items);

    var csv = this.convertToCSV(jsonObject);

    var exportedFilenmae = fileTitle + '.csv' || 'export.csv';

    var blob = new Blob(["\uFEFF", csv], { type: 'text/csv;charset=utf-8;' });
    if (navigator.msSaveBlob) { // IE 10+
        navigator.msSaveBlob(blob, exportedFilenmae);
    } else {
        var link = document.createElement("a");
        if (link.download !== undefined) { // feature detection
            // Browsers that support HTML5 download attribute
            var url = URL.createObjectURL(blob);
            link.setAttribute("href", url);
            link.setAttribute("download", exportedFilenmae);
            link.style.visibility = 'hidden';
            document.body.appendChild(link);
            link.click();
            document.body.removeChild(link);
        }
    }
}

/**
 * Convert sms list object to string
 * @param  {Object} objArray The list of sms items
 * @return {String}          The formatted string
 * @since    1.0.0
 */
function convertToCSV(objArray) {
    var array = typeof objArray != 'object' ? JSON.parse(objArray) : objArray;
    var str = '';
    for (var i = 0; i < array.length; i++) {
        var line = '';
        for (var index in array[i]) {
            if (line != '') line += ';'
            line += array[i][index];
        }
        str += line + '\r\n';
    }
    return str;
}
