<?php
/**
 * Defines a class that inherits from abstract class
 * Sms_Plugin_Com_Api_Base
 *
 * @package    Sms_Plugin_Com
 * @subpackage Sms_Plugin_Com/includes/api
 * @link       https://www.sms-plugin.com
 * @author     sms-plugin.com <info@sms-plugin.com>
 * @since 1.0.0
 */

require_once dirname( __FILE__ ) . '/class-sms-plugin-com-api-base.php';

/**
 * Class Sms_Plugin_Com_Api inherits from Sms_Plugin_Com_Api_Base
 * and it is a singleton.
 *
 * @package    Sms_Plugin_Com
 * @subpackage Sms_Plugin_Com/includes/api
 * @author     sms-plugin.com <info@sms-plugin.com>
 * @since 1.0.0
 */
class Sms_Plugin_Com_Api extends Sms_Plugin_Com_Api_Base {
	/**
	 * The plugin instance
	 *
	 * @var Sms_Plugin_Com_Api instance
	 * @since 1.0.0
	 */
	private static $instance;

	/**
	 * Constructor
	 *
	 * @since 1.0.0
	 * @access private
	 * @return void
	 */
	private function __construct() { }

	/**
	 * Function instance
	 * This method makes sure that only one instance of
	 * this class exists in memory at any given time.
	 *
	 * @since 1.0.0
	 * @access public
	 * @return Sms_Plugin_Com_Api
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) && ! ( self::$instance instanceof Sms_Plugin_Com_Api ) ) {
			self::$instance = new Sms_Plugin_Com_Api();
		}
		return self::$instance;
	}

	/**
	 * Action used to add the neccessary js at the footer in order
	 * to get the balance upon plugins_loaded hook
	 *
	 * @since 1.0.0
	 * @access public
	 */
	public function add_get_balance_js() {
		$params = array(
			'message_200' => __( 'Your remaining SMS balance', 'sms_plugin_com' ),
			'message_401' => __( 'Wrong Username/Password. Please check your credentials', 'sms_plugin_com' ),
			'message_500' => __( 'There has been an issue. Please try again later', 'sms_plugin_com' )
		);
		// Add balance script.
		$this->add_script( 'sms_plugin_com_get_balance', 'sms_plugin_com_get_balance', 'sms-plugin-com-get-balance.js', $params, true );
	}

	/**
	 * Action used to add get-key button js at the footer upon admin_init hook
	 *
	 * @since 1.0.0
	 * @access public
	 */
	public function add_get_key_js() {
		$params = array(
			'message_200' => __( 'Your API key {$} has been fetched', 'sms_plugin_com' ),
			'message_401' => __( 'Wrong Username/Password. Please check your credentials', 'sms_plugin_com' ),
			'message_500' => __( 'There has been an issue. Please try again later', 'sms_plugin_com' ),
			'username_missing' => __( 'Userame is required', 'sms_plugin_com' ),
			'password_missing' => __( 'Password is required', 'sms_plugin_com' )
		);
		// Add get_key script.
		$this->add_script( 'sms_plugin_com_get_key', 'sms_plugin_com_get_key', 'sms-plugin-com-get-key.js', $params, true );
	}

	/**
	 * Action used to add reset-key button js at the footer upon admin_init hook
	 *
	 * @since 1.0.0
	 * @access public
	 */
	public function add_reset_key_js() {
		$params = array(
			'message_200' => __( 'Your API key has been reset to {$} and saved', 'sms_plugin_com' ),
			'message_401' => __( 'Invalid API key. Please check your credentials', 'sms_plugin_com' ),
			'message_500' => __( 'There has been an issue. Please try again later', 'sms_plugin_com' ),
			'username_missing' => __( 'Userame is required', 'sms_plugin_com' ),
			'password_missing' => __( 'Password is required', 'sms_plugin_com' )
		);
		// Add reset_key script.
		$this->add_script( 'sms_plugin_com_reset_key', 'sms_plugin_com_reset_key', 'sms-plugin-com-reset-key.js', $params, true );
	}

	/**
	 * Action used to add the neccessary js at the footer in order
	 * to send sms
	 *
	 * @since 1.0.0
	 * @access public
	 */
	public function add_send_sms_js() {
		$params = array(
			'message_200' => __( 'Your remaining SMS balance', 'sms_plugin_com' ),
			'message_401' => __( 'Wrong Username/Password. Please check your credentials', 'sms_plugin_com' ),
			'message_402' => __( 'Insufficient balance. Please charge your account', 'sms_plugin_com' ),
			'message_500' => __( 'There has been an issue. Please try again later', 'sms_plugin_com' ),
			'sender_missing' => __( 'Sender Name is required', 'sms_plugin_com' ),
			'recipient_missing' => __( 'Recipient Mobile is required', 'sms_plugin_com' ),
			'message_missing' => __( 'SMS text is required', 'sms_plugin_com' ),
			'message_too_long' => __( 'SMS text must be greater than 0 and equal or less than 918 characters', 'sms_plugin_com' )
		);
		// Add send_sms script.
		$this->add_script( 'sms_plugin_com_send_sms', 'sms_plugin_com_send_sms', 'sms-plugin-com-send-sms.js', $params, true );
	}

	/**
	 * Action used to add the neccessary js at the footer in order
	 * to add sms counter functionality in actions tab
	 *
	 * @since 1.0.0
	 * @access public
	 */
	public function add_actions_js() {
		// Add actions script.
		$this->add_script( 'sms_plugin_com_actions', 'sms_plugin_com_actions', 'sms-plugin-com-actions.js', [], true );
	}

	/**
	 * Action used to add the neccessary js at the footer in order
	 * to add history table functionality
	 *
	 * @since 1.0.0
	 * @access public
	 */
	public function add_history_js() {
		$params = array(
			'message_200' => __( 'Your sms csv file is ready', 'sms_plugin_com' ),
			'message_204' => __( 'No sms to export in csv file!', 'sms_plugin_com' )
		);
		// Add history script.
		$this->add_script( 'sms_plugin_com_history', 'sms_plugin_com_history', 'sms-plugin-com-history.js', $params, true );
	}

	/**
	 * Function used to enqueue script
	 *
	 * @since 1.0.0
	 * @param string $handle    [the unique identifier].
	 * @param string $variable  [The name of the variable which will contain the data].
	 * @param string $source    [the name of the javascript file].
	 * @param array  $params    [the parameters to pass to the script].
	 * @param bool   $in_footer [whether the script will be inserted to the footer].
	 */
	public function add_script( $handle, $variable, $source, $params = null, $in_footer ) {
		// Enqueue the script.
		wp_enqueue_script(
			$handle, plugin_dir_url( __FILE__ ) . 'js/' . $source,
			array( 'jquery' ), SMS_PLUGIN_COM_VERSION, $in_footer
		);

		// Get current page protocol.
		$protocol = isset( $_SERVER['HTTPS'] ) ? 'https://' : 'http://';

		/**
	 * Output admin-ajax.php URL with same protocol as current page
	 * and messages to be displayed.
	 */
		if(null !== $params) {
			$params = array(
				'url' => admin_url( 'admin-ajax.php', $protocol ),
				'nonce' => wp_create_nonce( $handle ),
			) + $params;
			wp_localize_script( $handle, $variable, $params );
		}
	}
}
