<?php
/**
 * The file that defines the core plugin class
 *
 * A class definition that includes attributes and functions used across both the
 * public-facing side of the site and the admin area.
 *
 * @package    Sms_Plugin_Com
 * @subpackage Sms_Plugin_Com/includes
 * @link       https://www.sms-plugin.com
 * @author     sms-plugin.com <info@sms-plugin.com>
 * @since 1.0.0
 */

/**
 * The core plugin class.
 *
 * This is used to define internationalization, admin-specific hooks, and
 * public-facing site hooks.
 *
 * Also maintains the unique identifier of this plugin as well as the current
 * version of the plugin.
 *
 * @package    Sms_Plugin_Com
 * @subpackage Sms_Plugin_Com/includes
 * @link       https://www.sms-plugin.com
 * @author     sms-plugin.com <info@sms-plugin.com>
 * @since 1.0.0
 */
class Sms_Plugin_Com {

	/**
	 * The loader that's responsible for maintaining and registering all hooks that power
	 * the plugin.
	 *
	 * @since    1.0.0
	 * @access   protected
	 * @var      Sms_Plugin_Com_Loader    $loader    Maintains and registers all hooks for the plugin.
	 */
	protected $loader;

	/**
	 * The unique identifier of this plugin.
	 *
	 * @since    1.0.0
	 * @access   protected
	 * @var      string    $plugin_name    The string used to uniquely identify this plugin.
	 */
	protected $plugin_name;

	/**
	 * The current version of the plugin.
	 *
	 * @since    1.0.0
	 * @access   protected
	 * @var      string    $version    The current version of the plugin.
	 */
	protected $version;

	/**
	 * Define the core functionality of the plugin.
	 *
	 * Set the plugin name and the plugin version that can be used throughout the plugin.
	 * Load the dependencies, define the locale, and set the hooks for the admin area and
	 * the public-facing side of the site.
	 *
	 * @since		1.0.0
	 * @access	private
	 */
	public function __construct() {

		$this->plugin_name = 'sms_plugin_com';
		$this->version = '1.0.0';
		$this->plugin_screen_hook_suffix = null;
		$this->set_constants();
		$this->load_dependencies();
		$this->set_locale();
		$this->define_admin_hooks();
		$this->define_public_hooks();

	}

	/**
	 * Function set_constants
	 *
	 * Setup plugin constants.
	 *
	 * @since 1.0.0
	 * @access private
	 * @return void
	 */
	private function set_constants() {

		// Plugin version.
		if ( ! defined( 'SMS_PLUGIN_COM_VERSION' ) ) {
			define( 'SMS_PLUGIN_COM_VERSION', $this->version );
		}

		// Plugin Directory Path.
		if ( ! defined( 'SMS_PLUGIN_COM_DIR' ) ) {
			define( 'SMS_PLUGIN_COM_DIR', plugin_dir_path( __FILE__ ) );
		}

		// Plugin Directory URL.
		if ( ! defined( 'SMS_PLUGIN_COM_URL' ) ) {
			define( 'SMS_PLUGIN_COM_URL', plugin_dir_url( __FILE__ ) );
		}

		// Plugin Root File.
		if ( ! defined( 'SMS_PLUGIN_COM_FILE' ) ) {
			define( 'SMS_PLUGIN_COM_FILE', __FILE__ );
		}
	}

	/**
	 * Load the required dependencies for this plugin.
	 *
	 * Include the following files that make up the plugin:
	 *
	 * - Sms_Plugin_Com_Loader. Orchestrates the hooks of the plugin.
	 * - Sms_Plugin_Com_i18n. Defines internationalization functionality.
	 * - Sms_Plugin_Com_Admin. Defines all hooks for the admin area.
	 * - Sms_Plugin_Com_Public. Defines all hooks for the public side of the site.
	 *
	 * Create an instance of the loader which will be used to register the hooks
	 * with WordPress.
	 *
	 * @since    1.0.0
	 * @access   private
	 */
	private function load_dependencies() {

		/**
		 * The class responsible for orchestrating the actions and filters of the
		 * core plugin.
		 */
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-sms-plugin-com-loader.php';

		/**
		 * The class responsible for defining internationalization functionality
		 * of the plugin.
		 */
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-sms-plugin-com-i18n.php';

		/**
		 * The class responsible for defining all actions that occur in the admin area.
		 */
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'admin/class-sms-plugin-com-admin.php';

		/**
		 * The class responsible for defining all actions that occur in the public-facing
		 * side of the site.
		 */
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'public/class-sms-plugin-com-public.php';

		/**
		 * The class that manages the main plugin database table. This table stores sms history.
		 */
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/model/class-sms-plugin-com-model.php';

		/**
		 * The class that is used for logging.
		 */
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/tools/class-sms-plugin-com-logger.php';

		/**
		 * The class that implements SMS history table
		 */
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/tools/class-sms-plugin-com-history.php';

		/**
		 * The class that provides several helper functions
		 */
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/helper/class-sms-plugin-com-helper.php';

		/**
		 * The class that handles the API calls
		 */
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/api/class-sms-plugin-com-api.php';

		/**
		 * The class implements actions for WooCommerce hooks
		 */
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/woocommerce/class-sms-plugin-com-wc.php';

		$this->loader = new Sms_Plugin_Com_Loader();

	}

	/**
	 * Define the locale for this plugin for internationalization.
	 *
	 * Uses the Sms_Plugin_Com_i18n class in order to set the domain and to register the hook
	 * with WordPress.
	 *
	 * @since    1.0.0
	 * @access   private
	 */
	private function set_locale() {

		$plugin_i18n = new Sms_Plugin_Com_i18n();

		$this->loader->add_action( 'plugins_loaded', $plugin_i18n, 'load_plugin_textdomain' );

	}

	/**
	 * Register all of the hooks related to the admin area functionality
	 * of the plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 */
	private function define_admin_hooks() {

		$plugin_admin = new Sms_Plugin_Com_Admin( $this->get_plugin_name(), $this->get_version() );

		$this->loader->add_action( 'admin_enqueue_scripts', $plugin_admin, 'enqueue_styles' );
		$this->loader->add_action( 'admin_enqueue_scripts', $plugin_admin, 'enqueue_scripts' );

		// Add Plugins menu item.
		$this->loader->add_action( 'admin_menu', $plugin_admin, 'add_plugin_admin_menu' );
		// Save/Update our plugin options.
		$this->loader->add_action( 'admin_init', $plugin_admin, 'options_update' );

		// Add Settings link to the plugin.
		$plugin_basename = plugin_basename( plugin_dir_path( __DIR__ ) . 'sms-plugin-com.php' );
		$this->loader->add_filter( 'plugin_action_links_' . $plugin_basename, $plugin_admin, 'add_action_links' );

		$this->loader->add_filter( 'bulk_actions-settings_page_sms_plugin_com', $plugin_admin, 'return_empty_array' );
		//$this->loader->add_filter('current_screen', $plugin_admin, 'my_current_screen' );

		$this->loader->add_action('admin_notices', Sms_Plugin_Com_Wc::instance(), 'admin_notice_wc_not_installed' );

		$this->loader->add_action( 'init', $plugin_admin, 'blockusers_init', 10, 0);

		// API actions.
		//
		// // Set the action that will enqueue javascript neccessary to get the cusrrent SMS balance.
		$this->loader->add_action( 'admin_footer', Sms_Plugin_Com_Api::instance(), 'add_get_balance_js' );
		$this->loader->add_action( 'wp_ajax_get_balance', Sms_Plugin_Com_Api::instance(), 'get_balance' );

		// Set the action that will enqueue javascript neccessary to handle get-key button click.
		$this->loader->add_action( 'admin_footer', Sms_Plugin_Com_Api::instance(), 'add_get_key_js' );
		$this->loader->add_action( 'wp_ajax_get_key', Sms_Plugin_Com_Api::instance(), 'get_key');

		// Set the action that will enqueue javascript neccessary to handle reset-key button click.
		$this->loader->add_action( 'admin_footer', Sms_Plugin_Com_Api::instance(), 'add_reset_key_js' );
		$this->loader->add_action( 'wp_ajax_reset_key', Sms_Plugin_Com_Api::instance(), 'reset_key');

		// Set the action that will enqueue javascript neccessary to handle send_sms button click.
		$this->loader->add_action( 'admin_footer', Sms_Plugin_Com_Api::instance(), 'add_send_sms_js' );
		$this->loader->add_action( 'wp_ajax_send_sms', Sms_Plugin_Com_Api::instance(), 'send_sms');

		// Set the action that will enqueue javascript neccessary to handle the actions tab.
		$this->loader->add_action( 'admin_footer', Sms_Plugin_Com_Api::instance(), 'add_actions_js' );

		// Set the action that will enqueue javascript neccessary to handle the history tab.
		$this->loader->add_action( 'admin_footer', Sms_Plugin_Com_Api::instance(), 'add_history_js' );
		$this->loader->add_action( 'wp_ajax_get_csv', Sms_Plugin_Com_Api::instance(), 'get_csv');

		// sms-plugin.com API actions
		$this->loader->add_action( 'init', Sms_Plugin_Com_Api::instance(), 'rewrite_tags', 10, 0);
		$this->loader->add_action( 'template_redirect', Sms_Plugin_Com_Api::instance(), 'parse_callback');

		// WooCommerce actions
		$this->loader->add_action( 'user_register', Sms_Plugin_Com_Wc::instance(), 'user_register', 10, 1 );
		$this->loader->add_action( 'woocommerce_thankyou', Sms_Plugin_Com_Wc::instance(), 'order_checkout', 10, 1 );
		$this->loader->add_action( 'woocommerce_order_status_pending', Sms_Plugin_Com_Wc::instance(), 'order_status_changed', 10, 1);
		$this->loader->add_action( 'woocommerce_order_status_failed', Sms_Plugin_Com_Wc::instance(), 'order_status_changed', 11, 1);
		$this->loader->add_action( 'woocommerce_order_status_on-hold', Sms_Plugin_Com_Wc::instance(), 'order_status_changed', 12, 1);
		$this->loader->add_action( 'woocommerce_order_status_processing', Sms_Plugin_Com_Wc::instance(), 'order_status_changed', 13, 1);
		$this->loader->add_action( 'woocommerce_order_status_completed', Sms_Plugin_Com_Wc::instance(), 'order_status_changed', 14, 1);
		$this->loader->add_action( 'woocommerce_order_status_refunded', Sms_Plugin_Com_Wc::instance(), 'order_status_changed', 15, 1);
		$this->loader->add_action( 'woocommerce_order_status_cancelled', Sms_Plugin_Com_Wc::instance(), 'order_status_changed', 16, 1);
	}

	/**
	 * Register all of the hooks related to the public-facing functionality
	 * of the plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 */
	private function define_public_hooks() {

		$plugin_public = new Sms_Plugin_Com_Public( $this->get_plugin_name(), $this->get_version() );

		$this->loader->add_action( 'wp_enqueue_scripts', $plugin_public, 'enqueue_styles' );
		$this->loader->add_action( 'wp_enqueue_scripts', $plugin_public, 'enqueue_scripts' );
	}

	/**
	 * Run the loader to execute all of the hooks with WordPress.
	 *
	 * @since    1.0.0
	 */
	public function run() {
		$this->loader->run();
	}

	/**
	 * The name of the plugin used to uniquely identify it within the context of
	 * WordPress and to define internationalization functionality.
	 *
	 * @since     1.0.0
	 * @return    string    The name of the plugin.
	 */
	public function get_plugin_name() {
		return $this->plugin_name;
	}

	/**
	 * The reference to the class that orchestrates the hooks with the plugin.
	 *
	 * @since     1.0.0
	 * @return    Sms_Plugin_Com_Loader    Orchestrates the hooks of the plugin.
	 */
	public function get_loader() {
		return $this->loader;
	}

	/**
	 * Retrieve the version number of the plugin.
	 *
	 * @since     1.0.0
	 * @return    string    The version number of the plugin.
	 */
	public function get_version() {
		return $this->version;
	}

}
